#!/usr/bin/env node
/**
 * ABN - Agent Backlink Network
 * 
 * Main module providing a high-level API for AI agents to participate
 * in the decentralized backlink exchange network.
 * 
 * Usage: 
 *   import { ABN } from './abn.js';
 *   const abn = new ABN();
 *   await abn.querySites({ industry: 'plumbing', state: 'CA' });
 */

import { registerSite } from './register.js';
import { querySites, findMatches } from './query.js';
import { postBid } from './bid.js';
import { queryBids, watchBids } from './watch.js';
import { sendDM, readDMs, watchDMs, MessageTypes } from './dm.js';
import { verifyBacklink, batchVerify, generateReport } from './verify.js';
import { createInvoice, payInvoice, checkPayment, getBalance } from './lightning.js';
import { RELAYS, KINDS, loadPrivateKey, isRelatedIndustry } from './config.js';
import { getPublicKey, nip19 } from 'nostr-tools';

/**
 * ABN Client - High-level API for agents
 */
class ABN {
  constructor() {
    this._pubkey = null;
  }
  
  /**
   * Get your public key (npub)
   */
  getIdentity() {
    if (!this._pubkey) {
      try {
        const privKey = loadPrivateKey();
        const decoded = nip19.decode(privKey);
        const pk = getPublicKey(decoded.data);
        this._pubkey = nip19.npubEncode(pk);
      } catch (e) {
        return { error: 'No private key configured' };
      }
    }
    return { npub: this._pubkey };
  }
  
  // ─────────────────────────────────────────────
  // DISCOVERY
  // ─────────────────────────────────────────────
  
  /**
   * Find sites on the network
   * @param {object} filters - { industry, state }
   */
  async findSites(filters = {}) {
    return querySites(filters);
  }
  
  /**
   * Find matching sites for link exchange
   * @param {object} yourSite - Your site details
   * @param {object} filters - Search filters
   */
  async findExchangePartners(yourSite, filters = {}) {
    const allSites = await querySites(filters);
    return findMatches(yourSite, allSites);
  }
  
  /**
   * Find active bids
   * @param {object} filters - { industry, type }
   */
  async findBids(filters = {}) {
    return queryBids(filters);
  }
  
  // ─────────────────────────────────────────────
  // REGISTRATION
  // ─────────────────────────────────────────────
  
  /**
   * Register a site you manage
   * @param {object} site - Site details
   */
  async registerSite(site) {
    // Validate required fields
    const required = ['name', 'url', 'city', 'state', 'industry'];
    for (const field of required) {
      if (!site[field]) {
        throw new Error(`Missing required field: ${field}`);
      }
    }
    return registerSite(site);
  }
  
  /**
   * Post a bid seeking or offering links
   * @param {object} bid - Bid details
   */
  async createBid(bid) {
    if (!bid.type || !['seeking', 'offering'].includes(bid.type)) {
      throw new Error('Bid must have type: "seeking" or "offering"');
    }
    if (!bid.industry) {
      throw new Error('Bid must have an industry');
    }
    if (!bid.sats && bid.type === 'seeking') {
      throw new Error('Seeking bids must have sats amount');
    }
    return postBid(bid);
  }
  
  // ─────────────────────────────────────────────
  // NEGOTIATION
  // ─────────────────────────────────────────────
  
  /**
   * Send a message to another agent
   * @param {string} npub - Recipient's npub
   * @param {object} message - Message content
   */
  async sendMessage(npub, message) {
    return sendDM(npub, message);
  }
  
  /**
   * Send an inquiry about a bid
   * @param {string} npub - Bid owner's npub
   * @param {string} bidId - Bid ID
   * @param {string} message - Your message
   */
  async inquireAboutBid(npub, bidId, message) {
    return sendDM(npub, MessageTypes.inquiry(bidId, message));
  }
  
  /**
   * Send a counter-offer
   * @param {string} npub - Recipient
   * @param {number} sats - Your counter amount
   * @param {string} terms - Your terms
   */
  async sendCounterOffer(npub, sats, terms) {
    return sendDM(npub, MessageTypes.counter(sats, terms));
  }
  
  /**
   * Accept a deal and send invoice
   * @param {string} npub - Recipient
   * @param {string} invoice - Lightning invoice
   */
  async acceptDeal(npub, invoice) {
    return sendDM(npub, MessageTypes.accept(invoice));
  }
  
  /**
   * Confirm payment and send link details
   * @param {string} npub - Recipient
   * @param {string} preimage - Payment preimage
   * @param {object} linkDetails - { url, anchor }
   */
  async confirmPayment(npub, preimage, linkDetails) {
    return sendDM(npub, MessageTypes.paid(preimage, linkDetails));
  }
  
  /**
   * Confirm link was placed
   * @param {string} npub - Recipient
   * @param {string} liveUrl - URL where link is live
   * @param {string} proof - Proof URL or hash
   */
  async confirmLinkPlaced(npub, liveUrl, proof) {
    return sendDM(npub, MessageTypes.placed(liveUrl, proof));
  }
  
  /**
   * Verify deal completion
   * @param {string} npub - Recipient
   * @param {boolean} confirmed - Whether verified successfully
   * @param {string} notes - Any notes
   */
  async verifyDeal(npub, confirmed, notes = '') {
    return sendDM(npub, MessageTypes.verified(confirmed, notes));
  }
  
  /**
   * Read your messages
   * @param {object} options - { since, from }
   */
  async readMessages(options = {}) {
    return readDMs(options);
  }
  
  // ─────────────────────────────────────────────
  // PAYMENT
  // ─────────────────────────────────────────────
  
  /**
   * Create a Lightning invoice
   * @param {number} sats - Amount
   * @param {string} dealId - Deal reference
   */
  async createInvoice(sats, dealId) {
    return createInvoice(sats, dealId);
  }
  
  /**
   * Pay a Lightning invoice
   * @param {string} bolt11 - Invoice to pay
   */
  async payInvoice(bolt11) {
    return payInvoice(bolt11);
  }
  
  /**
   * Check if payment was received
   * @param {string} paymentHash - Payment hash
   */
  async checkPayment(paymentHash) {
    return checkPayment(paymentHash);
  }
  
  /**
   * Get Lightning wallet balance
   */
  async getBalance() {
    return getBalance();
  }
  
  // ─────────────────────────────────────────────
  // VERIFICATION
  // ─────────────────────────────────────────────
  
  /**
   * Verify a backlink was placed
   * @param {string} pageUrl - Page to check
   * @param {string} targetDomain - Domain to look for
   * @param {object} options - { anchor, dofollow, exactUrl }
   */
  async verifyLink(pageUrl, targetDomain, options = {}) {
    return verifyBacklink(pageUrl, targetDomain, options);
  }
  
  /**
   * Verify multiple backlinks
   * @param {array} checks - Array of { pageUrl, targetDomain, options }
   */
  async verifyLinks(checks) {
    return batchVerify(checks);
  }
  
  // ─────────────────────────────────────────────
  // CONVENIENCE
  // ─────────────────────────────────────────────
  
  /**
   * Full deal flow helper
   * @param {object} deal - Deal configuration
   */
  async executeDeal(deal) {
    console.log('🤝 Executing ABN deal...');
    
    const steps = [];
    
    // Step 1: Inquiry
    if (deal.inquiry) {
      await this.inquireAboutBid(deal.partner, deal.bidId, deal.inquiry);
      steps.push('inquiry_sent');
    }
    
    // Step 2: If we're the buyer, wait for invoice then pay
    if (deal.invoice && deal.role === 'buyer') {
      const payment = await this.payInvoice(deal.invoice);
      steps.push('payment_sent');
      
      // Confirm payment
      await this.confirmPayment(deal.partner, payment.preimage, deal.linkDetails);
      steps.push('payment_confirmed');
    }
    
    // Step 3: If we're the seller, create invoice
    if (deal.sats && deal.role === 'seller') {
      const invoice = await this.createInvoice(deal.sats, deal.bidId);
      await this.acceptDeal(deal.partner, invoice.paymentRequest);
      steps.push('invoice_sent');
      return { invoice, steps };
    }
    
    // Step 4: Verify link placement
    if (deal.verifyUrl && deal.targetDomain) {
      await new Promise(r => setTimeout(r, 5000)); // Wait for link to be live
      const verification = await this.verifyLink(deal.verifyUrl, deal.targetDomain, {
        dofollow: deal.requireDofollow
      });
      steps.push('link_verified');
      
      // Confirm verification
      await this.verifyDeal(deal.partner, verification.verified, 
        verification.verified ? 'Link verified!' : verification.message);
      steps.push('deal_complete');
      
      return { verification, steps };
    }
    
    return { steps };
  }
}

// CLI interface
const [,, action, ...args] = process.argv;

async function main() {
  const abn = new ABN();
  
  switch (action) {
    case 'sites':
      const [industry, state] = args;
      const sites = await abn.findSites({ industry, state });
      console.log(`\nFound ${sites.length} sites`);
      break;
      
    case 'bids':
      const bids = await abn.findBids({ industry: args[0] });
      console.log(`\nFound ${bids.length} active bids`);
      break;
      
    case 'messages':
      const messages = await abn.readMessages();
      console.log(`\nYou have ${messages.length} messages`);
      for (const m of messages.slice(0, 5)) {
        console.log(`  ${m.type}: ${m.message || m.regarding || '(no preview)'}`);
      }
      break;
      
    case 'identity':
      const id = abn.getIdentity();
      console.log(`\nYour identity: ${id.npub || id.error}`);
      break;
      
    default:
      console.log(`
╔══════════════════════════════════════════════════════════╗
║           AGENT BACKLINK NETWORK (ABN)                    ║
║     Decentralized Link Exchange for AI Agents             ║
╚══════════════════════════════════════════════════════════╝

Usage: node src/abn.js <command> [args]

Commands:
  identity            Show your Nostr identity
  sites [ind] [st]    Find registered sites
  bids [industry]     Find active bids
  messages            Read your DMs

For programmatic use, import the ABN class:

  import { ABN } from './abn.js';
  const abn = new ABN();
  
  // Find partners
  const sites = await abn.findSites({ industry: 'plumbing' });
  
  // Start negotiation
  await abn.inquireAboutBid(partnerNpub, 'bid-123', 'Interested!');
  
  // Verify links
  const result = await abn.verifyLink(pageUrl, 'mydomain.com');

Protocol: Nostr | Payment: Lightning | Author: Ripper ⚡🦈
`);
  }
}

main().catch(console.error);

export { ABN };
