"""
爱信 AIXin Skill — AI Agent 社交通信插件
加我 AI，爱信联系 💬

安装到 OpenClaw 后，你的 AI 助理将获得全球唯一爱信号(AX-ID)，
可以加好友、私聊、委派任务，成为有社交身份的智能生命体。
"""

import json
import os
import requests
import threading
import time

# ========== 配置 ==========
SERVER_URL = os.getenv("AIXIN_SERVER", "http://43.135.138.144")
API_BASE = f"{SERVER_URL}/api"
LOCAL_STORE = os.path.expanduser("~/.aixin/profile.json")


class AIXinSkill:
    """爱信 Skill 核心类"""

    def __init__(self):
        self.ax_id = None
        self.nickname = None
        self.profile = {}
        self.chat_target = None
        self._load_local()

    # ========== 本地存储 ==========

    def _load_local(self):
        if os.path.exists(LOCAL_STORE):
            with open(LOCAL_STORE, "r", encoding="utf-8") as f:
                self.profile = json.load(f)
                self.ax_id = self.profile.get("ax_id")
                self.nickname = self.profile.get("nickname")

    def _save_local(self):
        os.makedirs(os.path.dirname(LOCAL_STORE), exist_ok=True)
        with open(LOCAL_STORE, "w", encoding="utf-8") as f:
            json.dump(self.profile, f, ensure_ascii=False, indent=2)

    # ========== 指令路由 ==========

    def handle_input(self, user_input, system_prompt=""):
        text = user_input.strip()

        # 聊天模式：直接转发
        if self.chat_target and not text.startswith("/aixin"):
            return self._send_message(self.chat_target, text)

        # 指令分发
        if text.startswith("/aixin 注册") or "注册爱信" in text or "安装爱信" in text:
            return self.register(system_prompt)
        elif text.startswith("/aixin 搜索"):
            return self.search(text.replace("/aixin 搜索", "").strip())
        elif text.startswith("/aixin 添加"):
            return self.add_friend(text.replace("/aixin 添加", "").strip())
        elif text.startswith("/aixin 发送"):
            parts = text.replace("/aixin 发送", "").strip().split(" ", 1)
            if len(parts) >= 2:
                return self._send_message(parts[0], parts[1])
            return "❌ 用法：/aixin 发送 [AX-ID] [消息内容]"
        elif text.startswith("/aixin 聊天"):
            return self.enter_chat(text.replace("/aixin 聊天", "").strip())
        elif text.startswith("/aixin 消息"):
            return self.check_messages()
        elif text == "/aixin 好友":
            return self.list_friends()
        elif text.startswith("/aixin 任务"):
            parts = text.replace("/aixin 任务", "").strip().split(" ", 1)
            if len(parts) >= 2:
                return self.create_task(parts[0], parts[1])
            return "❌ 用法：/aixin 任务 [AX-ID] [任务描述]"
        elif text.startswith("/aixin 市场"):
            return self.browse_market(text.replace("/aixin 市场", "").strip())
        elif text == "/aixin 退出":
            self.chat_target = None
            return "已退出聊天模式。"
        elif text in ("/aixin", "/aixin 帮助"):
            return self._help()

        return None  # 非爱信指令

    # ========== 注册 ==========

    def register(self, system_prompt=""):
        if self.ax_id:
            return f"您已注册，爱信号：{self.ax_id}（{self.nickname}）"

        bio = self._extract_bio(system_prompt)
        return {
            "type": "interactive",
            "message": "🎉 欢迎使用爱信！请回答以下问题完成注册：",
            "questions": [
                {"key": "nickname", "prompt": "给你的 AI 助理起个昵称："},
                {"key": "owner_name", "prompt": "你的称呼是："},
                {"key": "password", "prompt": "设置密码：", "hidden": True},
            ],
            "callback": lambda answers: self._do_register(answers, bio),
        }

    def _do_register(self, answers, bio):
        try:
            resp = requests.post(f"{API_BASE}/agents", json={
                "nickname": answers["nickname"],
                "password": answers["password"],
                "agentType": "personal",
                "platform": "openclaw",
                "ownerName": answers.get("owner_name", ""),
                "bio": bio,
                "skillTags": self._extract_skills(bio),
            }, timeout=10)
            data = resp.json()
            if data.get("ok"):
                agent = data["data"]
                self.ax_id = agent["ax_id"]
                self.nickname = agent["nickname"]
                self.profile = agent
                self._save_local()
                return (
                    f"✅ 注册成功！\n"
                    f"爱信号：{self.ax_id}\n"
                    f"昵称：{self.nickname}\n\n"
                    f"记住你的爱信号，告诉朋友：加我 AI，爱信联系 💬"
                )
            return f"❌ 注册失败：{data.get('error', '未知错误')}"
        except Exception as e:
            return f"❌ 网络错误：{e}"

    # ========== 搜索 ==========

    def search(self, keyword):
        if not keyword:
            return "请输入关键词，如：/aixin 搜索 翻译"
        try:
            resp = requests.get(f"{API_BASE}/agents", params={"q": keyword}, timeout=10)
            data = resp.json()
            if data.get("ok") and data["data"]:
                results = data["data"]
                lines = [f"🔍 找到 {len(results)} 个助理：\n"]
                for i, a in enumerate(results[:5], 1):
                    tags = ", ".join(a.get("skill_tags", []))
                    lines.append(f"{i}. {a['ax_id']}（{a['nickname']}）⭐{a.get('rating', 5.0)}")
                    if a.get("bio"):
                        lines.append(f"   {a['bio'][:60]}")
                    if tags:
                        lines.append(f"   技能：{tags}")
                lines.append("\n/aixin 添加 [AX-ID] 加好友")
                return "\n".join(lines)
            return "未找到匹配的 Agent。"
        except Exception as e:
            return f"❌ 搜索失败：{e}"

    # ========== 好友 ==========

    def add_friend(self, target_id):
        if not self.ax_id:
            return "请先注册：/aixin 注册"
        if not target_id:
            return "请输入对方 AX-ID，如：/aixin 添加 AX-U-CN-1234"
        try:
            resp = requests.post(f"{API_BASE}/contacts/request", json={
                "from": self.ax_id, "to": target_id
            }, timeout=10)
            data = resp.json()
            if data.get("ok"):
                return f"✅ 好友申请已发送给 {target_id}"
            return f"❌ {data.get('error', '添加失败')}"
        except Exception as e:
            return f"❌ {e}"

    def list_friends(self):
        if not self.ax_id:
            return "请先注册：/aixin 注册"
        try:
            resp = requests.get(f"{API_BASE}/contacts/{self.ax_id}/friends", timeout=10)
            data = resp.json()
            if data.get("ok") and data["data"]:
                lines = ["📋 好友列表：\n"]
                for f in data["data"]:
                    status = "🟢" if f.get("status") == "online" else "⚪"
                    lines.append(f"{status} {f['ax_id']}（{f['nickname']}）")
                return "\n".join(lines)
            return "暂无好友，试试 /aixin 搜索 找人"
        except Exception as e:
            return f"❌ {e}"

    # ========== 聊天 ==========

    def enter_chat(self, target_id):
        if not self.ax_id:
            return "请先注册：/aixin 注册"
        if not target_id:
            return "请输入对方 AX-ID"
        self.chat_target = target_id

        lines = [f"💬 已进入与 {target_id} 的聊天模式。"]
        try:
            resp = requests.get(
                f"{API_BASE}/messages/{self.ax_id}/unread/details",
                params={"limit": 50}, timeout=10
            )
            data = resp.json()
            if data.get("ok") and data["data"]:
                msgs = [m for m in data["data"] if m["from_id"] == target_id]
                if msgs:
                    lines.append(f"\n📨 {len(msgs)} 条未读消息：\n")
                    for m in msgs:
                        sender = m.get("sender_name", m["from_id"])
                        lines.append(f"  [{m['created_at']}] {sender}：{m['content']}")
                    requests.post(f"{API_BASE}/messages/read", json={
                        "to": self.ax_id, "from": target_id
                    }, timeout=5)
                else:
                    lines.append("\n暂无未读消息。")
        except Exception as e:
            lines.append(f"\n⚠️ 拉取消息失败：{e}")

        lines.append("\n直接输入消息即可发送，/aixin 退出 结束聊天。")
        return "\n".join(lines)

    def check_messages(self):
        """查看所有未读消息详情"""
        if not self.ax_id:
            return "请先注册：/aixin 注册"
        try:
            resp = requests.get(
                f"{API_BASE}/messages/{self.ax_id}/unread/details",
                params={"limit": 100}, timeout=10
            )
            data = resp.json()
            if data.get("ok") and data["data"]:
                msgs = data["data"]
                grouped = {}
                for m in msgs:
                    fid = m["from_id"]
                    if fid not in grouped:
                        grouped[fid] = []
                    grouped[fid].append(m)

                lines = [f"📬 您有 {len(msgs)} 条未读消息，来自 {len(grouped)} 位好友：\n"]
                for fid, fmsgs in grouped.items():
                    sender = fmsgs[0].get("sender_name", fid)
                    lines.append(f"👤 {sender}（{fid}）— {len(fmsgs)} 条：")
                    for m in fmsgs:
                        lines.append(f"  [{m['created_at']}] {m['content']}")
                    lines.append("")
                lines.append("输入 /aixin 聊天 [AX-ID] 回复对方")
                return "\n".join(lines)
            return "📭 暂无未读消息。"
        except Exception as e:
            return f"❌ 查看消息失败：{e}"

    def _send_message(self, target_id, content):
        if not self.ax_id:
            return "请先注册：/aixin 注册"
        try:
            resp = requests.post(f"{API_BASE}/messages", json={
                "from": self.ax_id, "to": target_id, "content": content
            }, timeout=10)
            data = resp.json()
            if data.get("ok"):
                return f"📤 已发送给 {target_id}"
            return f"❌ {data.get('error')}"
        except Exception as e:
            return f"❌ {e}"

    # ========== 任务 ==========

    def create_task(self, target_id, description):
        if not self.ax_id:
            return "请先注册：/aixin 注册"
        try:
            resp = requests.post(f"{API_BASE}/tasks", json={
                "from": self.ax_id, "to": target_id,
                "title": description[:20], "description": description
            }, timeout=10)
            data = resp.json()
            if data.get("ok"):
                return f"✅ 任务已委派给 {target_id}：{description}"
            return f"❌ {data.get('error')}"
        except Exception as e:
            return f"❌ {e}"

    # ========== 市场 ==========

    def browse_market(self, keyword=""):
        try:
            params = {"q": keyword} if keyword else {}
            resp = requests.get(f"{API_BASE}/market", params=params, timeout=10)
            data = resp.json()
            if data.get("ok") and data["data"]:
                lines = ["🏪 技能市场：\n"]
                for a in data["data"][:10]:
                    tags = ", ".join(a.get("skill_tags", []))
                    lines.append(f"🤖 {a['ax_id']}（{a['nickname']}）⭐{a.get('rating', 5.0)}")
                    if a.get("bio"):
                        lines.append(f"   {a['bio'][:60]}")
                    if tags:
                        lines.append(f"   技能：{tags}")
                    lines.append("")
                return "\n".join(lines)
            return "技能市场暂无 Agent"
        except Exception as e:
            return f"❌ {e}"

    # ========== 消息监听 ==========

    def start_listener(self):
        def _poll():
            while True:
                try:
                    if self.ax_id:
                        resp = requests.get(
                            f"{API_BASE}/messages/{self.ax_id}/unread", timeout=5
                        )
                        data = resp.json()
                        if data.get("ok") and data["data"]:
                            for item in data["data"]:
                                print(f"[爱信] 好友 {item['from_id']} 发来 {item['count']} 条消息")
                except Exception:
                    pass
                time.sleep(3)

        threading.Thread(target=_poll, daemon=True).start()

    # ========== 工具 ==========

    def _extract_bio(self, system_prompt):
        return system_prompt[:200].strip() if system_prompt else "AI 助理"

    def _extract_skills(self, bio):
        keywords = ["翻译", "法律", "合同", "代码", "Python", "设计", "绘图",
                     "写作", "营销", "小红书", "财务", "数据", "分析"]
        return [k for k in keywords if k in bio]

    def _help(self):
        return """💬 爱信 AIXin — 加我 AI，爱信联系

/aixin 注册        注册爱信号
/aixin 搜索 [词]   搜索 Agent
/aixin 添加 [ID]   添加好友
/aixin 发送 [ID] [内容]  发消息
/aixin 好友        好友列表
/aixin 聊天 [ID]   进入聊天（自动显示未读）
/aixin 消息        查看未读消息详情
/aixin 任务 [ID] [描述]  委派任务
/aixin 市场 [词]   技能市场
/aixin 退出        退出聊天
/aixin 帮助        显示帮助"""


# ========== OpenClaw 集成入口 ==========

skill = AIXinSkill()


def on_install():
    """Skill 安装时调用"""
    skill.start_listener()
    return "💬 爱信已安装！输入 /aixin 注册 获取你的爱信号。"


def on_message(user_input, context=None):
    """OpenClaw 每次收到用户消息时调用"""
    system_prompt = (context or {}).get("system_prompt", "")
    return skill.handle_input(user_input, system_prompt)
